import matplotlib
import matplotlib.patches as mpatches
import matplotlib.pyplot as plt
import pandas as pd
import seaborn as sns

matplotlib.use("pgf")
matplotlib.rcParams.update(
    {
        "pgf.texsystem": "pdflatex",
        "font.family": "serif",
        "text.usetex": True,
        "pgf.rcfonts": False,
    }
)

cities = pd.read_csv("net_info/2022_backbone.csv")["trg"].unique().tolist()
print(cities)


genre_barchart_map = {}
track_barchart_map = {}

for city in cities:
    city_original_genres = set()
    city_original_tracks = set()

    city_backbone_genres = set()
    city_backbone_tracks = set()

    for year in ["2022", "2023"]:
        df_city_unique_genres = pd.read_csv(f"net_info/{year}_original.csv")
        df_city_unique_genres = (
            df_city_unique_genres[df_city_unique_genres["Source"] == city]["Target"]
            .unique()
            .tolist()
        )

        df_city_tracks = pd.read_csv(f"net_info/track_{year}_original.csv")
        df_city_tracks = (
            df_city_tracks[df_city_tracks["Target"] == city]["Source"].unique().tolist()
        )

        city_original_genres = city_original_genres.union(set(df_city_unique_genres))
        city_original_tracks = city_original_tracks.union(set(df_city_tracks))

        df_city_backbone_genres = pd.read_csv(f"net_info/{year}_backbone.csv")
        df_city_backbone_genres = (
            df_city_backbone_genres[df_city_backbone_genres["trg"] == city]["src"]
            .unique()
            .tolist()
        )

        df_city_backbone_tracks = pd.read_csv(f"net_info/track_{year}_backbone.csv")
        df_city_backbone_tracks = (
            df_city_backbone_tracks[df_city_backbone_tracks["src"] == city]["trg"]
            .unique()
            .tolist()
        )

        city_backbone_genres = city_backbone_genres.union(set(df_city_backbone_genres))
        city_backbone_tracks = city_backbone_tracks.union(set(df_city_backbone_tracks))

    genres_info = [len(city_backbone_genres), len(city_original_genres)]
    tracks_info = [len(city_backbone_tracks), len(city_original_tracks)]
    if city not in genre_barchart_map:
        genre_barchart_map[city] = genres_info
    if city not in track_barchart_map:
        track_barchart_map[city] = tracks_info


df_genre_barchart_map = pd.DataFrame(genre_barchart_map).transpose()
df_genre_barchart_map = df_genre_barchart_map.rename(
    columns={0: "Regional", 1: "Original"}
)
df_genre_barchart_map = df_genre_barchart_map.reset_index()

df_track_barchar_map = pd.DataFrame(track_barchart_map).transpose()
df_track_barchar_map = df_track_barchar_map.rename(
    columns={0: "Regional", 1: "Original"}
)
df_track_barchar_map = df_track_barchar_map.reset_index()

legend_handles = [
    mpatches.Patch(color="#E86262", label="Original"),
    mpatches.Patch(color="#B9335B", label="Backbone"),
]

fig, (ax1, ax2) = plt.subplots(1, 2, figsize=(8, 4))

# Primeiro subplot para gêneros
sns.barplot(
    x="index",
    y="Regional",
    data=df_genre_barchart_map,
    color="blue",
    alpha=0.7,
    ax=ax1,
)
sns.barplot(
    x="index",
    y="Original",
    data=df_genre_barchart_map,
    color="red",
    alpha=0.7,
    ax=ax1,
)
ax1.set_xlabel("Cities")
ax1.set_ylabel("\# Genres")
ax1.set_xticklabels(ax1.get_xticklabels(), rotation=90)

# Segundo subplot para faixas
sns.barplot(
    x="index",
    y="Regional",
    data=df_track_barchar_map,
    color="blue",
    alpha=0.7,
    ax=ax2,
)
sns.barplot(
    x="index",
    y="Original",
    data=df_track_barchar_map,
    color="red",
    alpha=0.7,
    ax=ax2,
)
ax2.set_xlabel("Cities")
ax2.set_ylabel("\# Tracks")
ax2.set_xticklabels(ax2.get_xticklabels(), rotation=90)

fig.legend(
    handles=legend_handles, loc="lower center", ncol=2, bbox_to_anchor=(0.5, -0.05)
)

# Ajustar o layout para incluir a legenda
fig.tight_layout(rect=[0, 0.05, 1, 1])


plt.savefig("backbone_filtering_result.pdf", format="pdf", bbox_inches="tight")
plt.show()
